
// Un serveur express optimisé pour héberger le site en local ou en production
const express = require('express');
const path = require('path');
const compression = require('compression');
const fs = require('fs');
const app = express();
const port = process.env.PORT || 3000;

// Fonction pour vérifier si la requête devrait être compressée
const shouldCompress = (req, res) => {
  // Ne pas compresser pour les anciens navigateurs qui ne supportent pas bien la compression
  if (req.headers['x-no-compression']) {
    return false;
  }
  // Ne pas compresser les petits fichiers
  if (res._contentLength && res._contentLength < 1024) {
    return false;
  }
  return compression.filter(req, res);
};

// Compression pour améliorer les performances
app.use(compression({ 
  level: 6, // Niveau de compression optimal entre vitesse et taux
  filter: shouldCompress,
  threshold: 1024 // Ne compresser que les réponses plus grandes que 1KB
}));

// En-têtes pour la sécurité et le cache
app.use((req, res, next) => {
  // Sécurité de base
  res.setHeader('X-Content-Type-Options', 'nosniff');
  res.setHeader('X-XSS-Protection', '1; mode=block');
  res.setHeader('X-Frame-Options', 'SAMEORIGIN');
  res.setHeader('Strict-Transport-Security', 'max-age=31536000; includeSubDomains');
  res.setHeader('Referrer-Policy', 'no-referrer-when-downgrade');
  
  // CORS configuré pour être restrictif par défaut, à adapter selon vos besoins
  res.setHeader('Access-Control-Allow-Origin', '*');
  
  // Stratégie de cache adaptative selon le type de fichier
  if (req.url.match(/\.(css|js)$/)) {
    // Cache plus long pour les ressources statiques qui changent peu
    res.setHeader('Cache-Control', 'public, max-age=604800'); // 1 semaine
  } else if (req.url.match(/\.(jpg|jpeg|png|gif|svg|webp|ico|woff|woff2)$/)) {
    // Cache très long pour les images et polices
    res.setHeader('Cache-Control', 'public, max-age=2592000, immutable'); // 30 jours, immutable
  } else {
    // Pas de cache pour les documents HTML et API
    res.setHeader('Cache-Control', 'no-cache, no-store, must-revalidate');
  }
  next();
});

// Middleware pour comprimer les fichiers SVG/JSON à la volée
app.get('*.svg', (req, res, next) => {
  req.url = req.url + '.gz';
  res.set('Content-Encoding', 'gzip');
  res.set('Content-Type', 'image/svg+xml');
  next();
});

app.get('*.json', (req, res, next) => {
  req.url = req.url + '.gz';
  res.set('Content-Encoding', 'gzip');
  res.set('Content-Type', 'application/json');
  next();
});

// Servir les fichiers statiques du build avec un middleware pour les logs en dev
const staticOptions = {
  etag: true,
  lastModified: true,
  maxAge: '1d',
  immutable: true, // Pour les assets versionnés
  setHeaders: (res, path) => {
    if (path.endsWith('.html')) {
      // Les HTML ne doivent pas être mis en cache de façon agressive
      res.setHeader('Cache-Control', 'no-cache');
    }
  }
};

app.use(express.static(path.join(__dirname, 'dist'), staticOptions));

// Middleware pour gérer les requêtes non trouvées - renvoie index.html pour les routes SPA
app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, 'dist', 'index.html'));
});

// Démarrage du serveur
app.listen(port, () => {
  console.log(`\x1b[32m%s\x1b[0m`, `✅ Le site est accessible sur http://localhost:${port}`);
  console.log(`\x1b[36m%s\x1b[0m`, `ℹ️ Pour déployer sur un serveur, configurez votre serveur web (Apache/Nginx) pour pointer vers le dossier 'dist'`);
  
  // Info sur la taille du build
  try {
    const distPath = path.join(__dirname, 'dist');
    if (fs.existsSync(distPath)) {
      const totalSize = calculateFolderSize(distPath);
      console.log(`\x1b[36m%s\x1b[0m`, `ℹ️ Taille totale du build: ${formatBytes(totalSize)}`);
    }
  } catch (error) {
    console.error('Erreur lors du calcul de la taille du build:', error);
  }
});

// Fonction pour calculer la taille d'un dossier
function calculateFolderSize(folderPath) {
  let totalSize = 0;
  const files = fs.readdirSync(folderPath, { withFileTypes: true });
  
  for (const file of files) {
    const filePath = path.join(folderPath, file.name);
    
    if (file.isDirectory()) {
      totalSize += calculateFolderSize(filePath);
    } else {
      totalSize += fs.statSync(filePath).size;
    }
  }
  
  return totalSize;
}

// Fonction pour formater la taille en bytes
function formatBytes(bytes, decimals = 2) {
  if (bytes === 0) return '0 Bytes';
  
  const k = 1024;
  const dm = decimals < 0 ? 0 : decimals;
  const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
  
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  
  return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}

// Gestion propre de l'arrêt du serveur
process.on('SIGTERM', shutDown);
process.on('SIGINT', shutDown);

function shutDown() {
  console.log('Fermeture du serveur...');
  process.exit(0);
}
