
import React, { useEffect, useRef, useState } from 'react';
import mapboxgl from 'mapbox-gl';
import 'mapbox-gl/dist/mapbox-gl.css';
import { Card, CardContent } from '@/components/ui/card';
import { Separator } from '@/components/ui/separator';

// Définition des types pour nos partenaires
interface Partner {
  name: string;
  address: string;
  coordinates: [number, number]; // [longitude, latitude]
  description?: string;
}

// Coordonnées des partenaires
const partners: Partner[] = [
  {
    name: "GIP Ecofor",
    address: "42 rue Scheffer, 75116 Paris, France",
    coordinates: [2.2769, 48.8631],
    description: "Siège social"
  },
  {
    name: "INRAE",
    address: "147 rue de l'Université, 75007 Paris, France",
    coordinates: [2.3144, 48.8580],
    description: "Institut national de recherche pour l'agriculture, l'alimentation et l'environnement"
  },
  {
    name: "ONF",
    address: "2 avenue de Saint-Mandé, 75012 Paris, France",
    coordinates: [2.4120, 48.8454],
    description: "Office National des Forêts"
  },
  {
    name: "CIRAD",
    address: "42 rue Scheffer, 75116 Paris, France",
    coordinates: [2.2770, 48.8633],
    description: "Centre de coopération internationale en recherche agronomique pour le développement"
  }
];

interface PartnerMapProps {
  className?: string;
}

const PartnerMap: React.FC<PartnerMapProps> = ({ className }) => {
  const mapContainer = useRef<HTMLDivElement>(null);
  const map = useRef<mapboxgl.Map | null>(null);
  const [mapboxToken, setMapboxToken] = useState<string>('');
  const [showTokenInput, setShowTokenInput] = useState(true);

  const initializeMap = (token: string) => {
    if (!mapContainer.current) return;
    
    mapboxgl.accessToken = token;
    
    // Création de la carte
    map.current = new mapboxgl.Map({
      container: mapContainer.current,
      style: 'mapbox://styles/mapbox/light-v11',
      center: [2.3522, 48.8566], // Paris
      zoom: 11
    });

    // Ajout des contrôles de navigation
    map.current.addControl(
      new mapboxgl.NavigationControl(),
      'top-right'
    );

    // Événement de chargement de la carte
    map.current.on('load', () => {
      // Ajouter les marqueurs pour chaque partenaire
      partners.forEach(partner => {
        // Création d'un élément DOM pour le marqueur
        const markerEl = document.createElement('div');
        markerEl.className = 'flex items-center justify-center w-6 h-6 bg-ecofor-dark-green rounded-full border-2 border-white shadow-md';
        
        // Ajout du marqueur sur la carte
        new mapboxgl.Marker(markerEl)
          .setLngLat(partner.coordinates)
          .setPopup(
            new mapboxgl.Popup({ offset: 25 })
              .setHTML(`
                <h3 class="text-sm font-bold">${partner.name}</h3>
                <p class="text-xs mt-1">${partner.address}</p>
                ${partner.description ? `<p class="text-xs text-gray-500 mt-1">${partner.description}</p>` : ''}
              `)
          )
          .addTo(map.current!);
      });
    });
  };

  // Gestion du jeton d'accès Mapbox
  const handleTokenSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (mapboxToken.trim()) {
      setShowTokenInput(false);
      initializeMap(mapboxToken);
      // Stockage local pour la session
      sessionStorage.setItem('mapboxToken', mapboxToken);
    }
  };

  useEffect(() => {
    // Vérification si un token est déjà stocké
    const savedToken = sessionStorage.getItem('mapboxToken');
    if (savedToken) {
      setMapboxToken(savedToken);
      setShowTokenInput(false);
      initializeMap(savedToken);
    }

    // Nettoyage lors du démontage du composant
    return () => {
      if (map.current) {
        map.current.remove();
      }
    };
  }, []);

  return (
    <Card className={`overflow-hidden ${className}`}>
      {showTokenInput ? (
        <CardContent className="p-6">
          <h3 className="text-lg font-semibold mb-4">Mapbox API Token Required</h3>
          <p className="text-sm text-muted-foreground mb-4">
            Pour afficher la carte des partenaires, veuillez saisir votre clé d'API Mapbox. 
            Vous pouvez obtenir une clé gratuite sur <a href="https://www.mapbox.com/" target="_blank" rel="noopener noreferrer" className="text-ecofor-dark-green hover:underline">mapbox.com</a>.
          </p>
          <form onSubmit={handleTokenSubmit} className="space-y-4">
            <input
              type="text"
              value={mapboxToken}
              onChange={(e) => setMapboxToken(e.target.value)}
              placeholder="Entrez votre clé API Mapbox"
              className="w-full p-3 rounded-md border border-ecofor-medium-blue/40 focus:outline-none focus:ring-2 focus:ring-ecofor-dark-green focus:border-transparent"
              required
            />
            <button 
              type="submit"
              className="w-full bg-ecofor-dark-green text-white px-4 py-2 rounded-md hover:bg-ecofor-dark-green/90 transition-colors"
            >
              Afficher la carte
            </button>
          </form>
        </CardContent>
      ) : (
        <>
          <div className="p-4 border-b">
            <h3 className="text-lg font-semibold">Localisation des partenaires</h3>
            <p className="text-sm text-muted-foreground">Découvrez où sont situés nos différents partenaires.</p>
          </div>
          <div className="h-96 w-full" ref={mapContainer} />
          <Separator />
          <div className="p-4 bg-gray-50">
            <h4 className="text-sm font-medium mb-2">Légende</h4>
            <div className="space-y-2">
              {partners.map((partner, index) => (
                <div key={index} className="flex items-center text-sm">
                  <div className="w-3 h-3 bg-ecofor-dark-green rounded-full mr-2"></div>
                  <span>{partner.name}</span>
                </div>
              ))}
            </div>
          </div>
        </>
      )}
    </Card>
  );
};

export default PartnerMap;
