
import { Link } from "react-router-dom";
import { Calendar, ArrowRight } from "lucide-react";
import SectionHeading from "../common/SectionHeading";
import { newsPosts, eventsPosts } from "@/data/newsExamples";

const FeaturedNews = () => {
  // Combiner et trier les actualités et événements par date (les plus récents d'abord)
  const combinedItems = [...newsPosts, ...eventsPosts]
    .sort((a, b) => new Date(b.date).getTime() - new Date(a.date).getTime())
    .slice(0, 3); // Prendre les 3 plus récents

  // Formater la date
  const formatDate = (dateString: string) => {
    const options: Intl.DateTimeFormatOptions = { 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    };
    return new Date(dateString).toLocaleDateString('fr-FR', options);
  };

  return (
    <section className="py-16 md:py-24 bg-gradient-to-b from-white to-forestt-primary/10">
      <div className="container">
        <SectionHeading 
          title="Actualités et Événements" 
          subtitle="Restez informés des dernières nouvelles, publications et événements de notre programme de recherche"
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {combinedItems.map((item, index) => (
            <article 
              key={item.id} 
              className="group flex flex-col h-full overflow-hidden rounded-xl border border-forestt-primary/20 bg-white shadow-sm transition-all hover:shadow-md animate-slide-up"
              style={{ animationDelay: `${index * 100 + 100}ms` }}
            >
              <div className="relative h-48 overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-t from-black/30 to-transparent z-10" />
                {item.type === "events" && (
                  <div className="absolute top-3 left-3 z-20 bg-forestt-highlight text-white px-2 py-1 rounded-full text-xs font-medium">
                    Événement
                  </div>
                )}
                <img 
                  src={item.image} 
                  alt={item.title} 
                  className="h-full w-full object-cover transition-transform duration-500 group-hover:scale-105"
                />
              </div>
              <div className="flex flex-col flex-grow p-6">
                <div className="flex items-center mb-2 text-sm text-muted-foreground">
                  <Calendar className="mr-2 h-4 w-4 text-forestt-secondary" />
                  <time dateTime={item.date}>{formatDate(item.date)}</time>
                </div>
                <h3 className="text-xl font-semibold mb-2 text-foreground group-hover:text-forestt-primary transition-colors">
                  {item.title}
                </h3>
                <p className="text-muted-foreground text-sm mt-2 mb-4 flex-grow">
                  {item.excerpt}
                </p>
                <Link 
                  to={`/news/${item.slug}`} 
                  className="inline-flex items-center text-forestt-primary font-medium transition-colors group-hover:text-forestt-secondary mt-auto"
                >
                  Lire la suite
                  <ArrowRight className="ml-1 h-4 w-4 transition-transform group-hover:translate-x-1" />
                </Link>
              </div>
            </article>
          ))}
        </div>

        <div className="flex justify-center mt-10">
          <Link 
            to="/news" 
            className="inline-flex items-center justify-center px-6 py-3 border border-forestt-primary text-forestt-primary bg-white rounded-md font-medium hover:bg-forestt-primary hover:text-white transition-colors"
          >
            Voir toutes les actualités
          </Link>
        </div>
      </div>
    </section>
  );
};

export default FeaturedNews;
