
import { Link } from "react-router-dom";
import { Calendar } from "lucide-react";
import { cn } from "@/lib/utils";
import OptimizedImage from "@/components/common/OptimizedImage";

export interface NewsItem {
  id: number;
  title: string;
  excerpt: string;
  date: string;
  type: string;
  image: string;
  slug: string;
  location?: string;
  tags?: string[];
}

interface NewsCardProps {
  item: NewsItem;
  index: number;
}

const NewsCard = ({ item, index }: NewsCardProps) => {
  return (
    <article 
      className={cn(
        "group rounded-xl overflow-hidden bg-white border border-gray-200 shadow-sm hover:shadow-md transition-all",
        "animate-slide-up flex flex-col h-full"
      )}
      style={{ animationDelay: `${index * 100}ms` }}
    >
      <Link to={`/news/${item.slug}`} className="flex flex-col h-full">
        <div className="relative h-48 overflow-hidden">
          <OptimizedImage
            src={item.image}
            alt={item.title}
            className="h-full w-full object-cover transition-transform duration-300 group-hover:scale-105"
          />
          <div className={cn(
            "absolute top-3 left-3 px-3 py-1 rounded-full text-xs font-medium",
            item.type === "events" 
              ? "bg-ecofor-turquoise/90 text-white" 
              : "bg-ecofor-light-blue text-ecofor-dark-green"
          )}>
            {item.type === "events" ? "Événement" : "Actualité"}
          </div>
        </div>
        
        <div className="p-6 flex flex-col flex-grow">
          <div className="flex items-center mb-3 text-sm text-muted-foreground">
            <Calendar className="h-4 w-4 mr-1" aria-hidden="true" />
            <time dateTime={item.date}>{item.date}</time>
          </div>
          
          <h3 className="text-xl font-bold mb-2 text-ecofor-dark-green group-hover:text-ecofor-turquoise transition-colors">
            {item.title}
          </h3>
          
          <p className="text-muted-foreground line-clamp-3 mb-4">
            {item.excerpt}
          </p>
          
          <div className="mt-auto">
            <span className="inline-flex items-center text-sm font-medium text-ecofor-turquoise group-hover:text-ecofor-dark-green transition-colors">
              Lire la suite
              <svg className="ml-1 h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
              </svg>
            </span>
          </div>
        </div>
      </Link>
    </article>
  );
};

export default NewsCard;
