
import { useEffect, useState } from 'react';
import { toast } from '@/components/ui/use-toast';
import { Skeleton } from '@/components/ui/skeleton';
import { getPartners } from '@/wordpress-config';
import { Button } from '@/components/ui/button';
import { ExternalLink } from 'lucide-react';
import { cn } from '@/lib/utils';

interface Partner {
  id: string | number;
  name: string;
  logo: string;
  website?: string;
  location?: string;
  type: string;
  description?: string;
}

interface WordPressPartnersListProps {
  partnerType?: string;
  showLocation?: boolean;
  showDescription?: boolean;
  maxPartners?: number;
}

const WordPressPartnersList = ({
  partnerType,
  showLocation = false,
  showDescription = false,
  maxPartners
}: WordPressPartnersListProps) => {
  const [partners, setPartners] = useState<Partner[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<boolean>(false);

  // Logos des partenaires pour fallback
  const partnerImages: Record<string, string> = {
    "AgroParisTech": "/lovable-uploads/5f65ecfb-bffd-4574-a470-b6a35e13e681.png",
    "CIRAD": "/lovable-uploads/51eb61a9-1c40-4e6f-b9a7-bd7b5ac797a8.png",
    "CNRS": "/lovable-uploads/162dea14-be0d-480d-9fa7-bcf25437696f.png",
    "ECOFOR": "/lovable-uploads/b022c4c0-5c09-47b8-aa17-061a272fa599.png",
    "EHESS": "/lovable-uploads/9738fd93-61b3-41e7-aebb-9412028b989f.png",
    "ENSFEA": "/lovable-uploads/c3c098c4-1632-40fd-b54a-3a792fa72b36.png",
    "Le Mans Université": "/lovable-uploads/b188469f-2ab9-40dc-a7dd-9c92a6318f05.png",
    "INRAE": "/lovable-uploads/fe646af5-39a4-49a4-b145-9bf577cf55d1.png",
    "ONF": "/lovable-uploads/5e0c5302-a2c2-498e-b7d8-a4d3fd0beab2.png",
    "Université de Montpellier": "/lovable-uploads/c8765bc5-2a53-4840-9b6d-ddfcf9cd8e16.png",
    "Université de Picardie Jules Verne": "/lovable-uploads/55db35ad-724f-406d-b66c-750ee49843a5.png",
    "Université de Lorraine": "/lovable-uploads/64454f46-282e-47b1-937a-49d06442a392.png",
    "Université Grenoble Alpes": "/lovable-uploads/e4e2bb1d-648e-4f96-8878-7d41756c6f1c.png",
    "Université de Pau et des Pays de l'Adour": "/lovable-uploads/e97f4c89-9839-4342-9b1d-b20de32135ad.png",
    "Université Toulouse 2 Jean Jaurès": "/lovable-uploads/dfd8ffe9-59c6-402b-892d-a80165db9c48.png"
  };

  // Fonction pour déterminer les classes spécifiques à certains logos
  const getLogoSpecificClasses = (partnerName: string) => {
    switch(partnerName) {
      case "ENSFEA":
      case "INRAE":
        // Réduire la taille des logos trop grands
        return "scale-90";
      case "Université Toulouse 2 Jean Jaurès":
        // Agrandir le logo trop petit
        return "scale-125";
      default:
        return "";
    }
  };

  useEffect(() => {
    const fetchPartners = async () => {
      setLoading(true);
      try {
        const data = await getPartners(partnerType);

        if (data && data.length > 0) {
          // Transformer les données WordPress en format pour le composant
          const formattedPartners: Partner[] = data.map((partner: any) => ({
            id: partner.id,
            name: partner.title.rendered,
            logo: partner.acf?.partner_logo?.url || 
                  (partner._embedded?.['wp:featuredmedia']?.[0]?.source_url || 
                  partnerImages[partner.title.rendered] || ''),
            website: partner.acf?.partner_website || '',
            location: partner.acf?.partner_location || '',
            type: partner._embedded?.['wp:term']?.[1]?.[0]?.name || 'Partenaire',
            description: partner.excerpt?.rendered || ''
          }));

          // Trier par ordre alphabétique
          formattedPartners.sort((a, b) => a.name.localeCompare(b.name));

          // Limiter le nombre de partenaires si demandé
          const limitedPartners = maxPartners ? formattedPartners.slice(0, maxPartners) : formattedPartners;
          setPartners(limitedPartners);
        } else {
          setError(true);
        }
      } catch (err) {
        console.error('Erreur lors du chargement des partenaires:', err);
        setError(true);
        toast({
          title: 'Erreur de chargement',
          description: 'Impossible de charger les partenaires depuis WordPress',
          variant: 'destructive'
        });
      } finally {
        setLoading(false);
      }
    };

    fetchPartners();
  }, [partnerType, maxPartners]);

  if (loading) {
    return (
      <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
        {Array(6).fill(0).map((_, index) => (
          <div key={index} className="bg-white p-4 rounded-lg shadow">
            <div className="h-24 flex items-center justify-center mb-4">
              <Skeleton className="h-16 w-full" />
            </div>
            <Skeleton className="h-6 w-3/4 mx-auto mb-2" />
            <Skeleton className="h-4 w-1/2 mx-auto" />
          </div>
        ))}
      </div>
    );
  }

  if (error || partners.length === 0) {
    return (
      <div className="text-center p-8 bg-muted rounded-lg">
        <h3 className="text-lg font-semibold mb-2">Aucun partenaire trouvé</h3>
        <p className="mb-4">Impossible de charger les données des partenaires.</p>
      </div>
    );
  }

  return (
    <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
      {partners.map((partner) => (
        <div key={partner.id} className="bg-white p-6 rounded-lg shadow-sm hover:shadow-md transition-shadow border">
          <div className="h-24 flex items-center justify-center mb-4">
            {partner.logo ? (
              <img 
                src={partner.logo} 
                alt={`Logo de ${partner.name}`} 
                className={cn(
                  "max-h-full max-w-full object-contain",
                  getLogoSpecificClasses(partner.name)
                )}
                loading="lazy"
              />
            ) : (
              <div className="h-16 w-full bg-gray-100 flex items-center justify-center text-gray-400 rounded">
                Logo non disponible
              </div>
            )}
          </div>

          <h3 className="text-lg font-semibold text-center mb-2" dangerouslySetInnerHTML={{ __html: partner.name }} />
          
          <div className="text-center text-sm text-forestt-secondary mb-4">
            {partner.type}
          </div>
          
          {showLocation && partner.location && (
            <div className="text-center text-sm text-gray-600 mb-3">
              {partner.location}
            </div>
          )}
          
          {showDescription && partner.description && (
            <div 
              className="text-sm text-gray-600 mb-4 line-clamp-3"
              dangerouslySetInnerHTML={{ __html: partner.description }}
            />
          )}
          
          {partner.website && (
            <div className="text-center mt-4">
              <Button 
                variant="outline" 
                size="sm" 
                className="inline-flex items-center gap-2"
                asChild
              >
                <a href={partner.website} target="_blank" rel="noopener noreferrer">
                  Visiter le site
                  <ExternalLink size={14} />
                </a>
              </Button>
            </div>
          )}
        </div>
      ))}
    </div>
  );
};

export default WordPressPartnersList;
