
import React from 'react';
import { Calendar, Download, ExternalLink, FileText } from "lucide-react";

interface Publication {
  id: number;
  title: string;
  authors: string;
  journal: string;
  date: string;
  abstract: string;
  tags: string[];
  pdfUrl?: string;
  doi?: string;
}

interface PublicationCardProps {
  publication: Publication;
}

const PublicationCard: React.FC<PublicationCardProps> = ({ publication }) => {
  return (
    <div className="bg-white rounded-lg shadow-md border border-ecofor-medium-blue/20 p-5 transition-shadow hover:shadow-lg">
      <div className="flex flex-col md:flex-row md:items-start gap-4">
        <div className="md:w-16 md:h-16 flex-shrink-0 bg-ecofor-light-blue rounded-lg flex items-center justify-center">
          <FileText className="w-8 h-8 text-ecofor-dark-green" />
        </div>
        <div className="flex-grow">
          <div className="flex flex-col md:flex-row md:items-center justify-between mb-2">
            <h3 className="text-xl font-semibold text-ecofor-dark-green mb-1 md:mb-0">{publication.title}</h3>
            <div className="flex items-center space-x-2">
              {publication.pdfUrl && (
                <a 
                  href={publication.pdfUrl}
                  className="inline-flex items-center px-3 py-1 bg-ecofor-dark-green text-white text-sm font-medium rounded hover:bg-ecofor-dark-green/90 transition-colors"
                  target="_blank"
                  rel="noopener noreferrer"
                >
                  <Download className="w-4 h-4 mr-1" /> PDF
                </a>
              )}
            </div>
          </div>
          
          <div className="flex flex-wrap items-center text-sm text-gray-600 mb-2 gap-x-4 gap-y-1">
            <span className="inline-flex items-center">
              <Calendar className="w-4 h-4 mr-1 text-ecofor-medium-blue" />
              {publication.date}
            </span>
            <span>
              <strong>Auteurs:</strong> {publication.authors}
            </span>
            <span>
              <strong>Journal:</strong> {publication.journal}
            </span>
            {publication.doi && (
              <a 
                href={`https://doi.org/${publication.doi}`}
                className="inline-flex items-center text-ecofor-turquoise hover:underline"
                target="_blank"
                rel="noopener noreferrer"
              >
                <ExternalLink className="w-3 h-3 mr-1" /> DOI: {publication.doi}
              </a>
            )}
          </div>
          
          <p className="text-gray-700 mb-3">
            {publication.abstract}
          </p>
          
          <div className="flex flex-wrap gap-2">
            {publication.tags.map((tag) => (
              <span 
                key={`${publication.id}-${tag}`}
                className="inline-block px-2 py-1 text-xs font-medium bg-ecofor-light-blue/40 text-ecofor-dark-green rounded-full"
              >
                {tag}
              </span>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
};

export default PublicationCard;
