
import React from 'react';
import { Calendar, Play, ExternalLink, Tag } from "lucide-react";
import YouTubeEmbed from "@/components/common/YouTubeEmbed";

interface Video {
  id: string;
  title: string;
  description: string;
  date: string;
  youtubeId: string;
  tags: string[];
  duration?: string;
  presenter?: string;
}

interface VideoCardProps {
  video: Video;
}

const VideoCard: React.FC<VideoCardProps> = ({ video }) => {
  return (
    <div className="bg-white rounded-lg overflow-hidden shadow-md hover:shadow-lg transition-shadow border border-ecofor-medium-blue/20">
      <div className="relative aspect-video group">
        <div className="absolute inset-0 flex items-center justify-center">
          <div className="bg-ecofor-dark-green/80 rounded-full p-4 text-white transform transition-transform group-hover:scale-110">
            <Play className="h-6 w-6" />
          </div>
        </div>
        <YouTubeEmbed videoId={video.youtubeId} title={video.title} />
      </div>
      <div className="p-5">
        <div className="flex items-center justify-between mb-2">
          <div className="flex items-center space-x-2">
            <Calendar className="h-4 w-4 text-ecofor-turquoise" />
            <span className="text-sm text-gray-600">{video.date}</span>
          </div>
          {video.duration && (
            <span className="text-xs font-medium bg-ecofor-light-blue/40 text-ecofor-dark-green px-2 py-1 rounded-full">
              {video.duration}
            </span>
          )}
        </div>
        
        <h2 className="text-xl font-bold mb-2 text-ecofor-dark-green">{video.title}</h2>
        
        {video.presenter && (
          <p className="text-sm text-ecofor-turquoise mb-2">
            Présenté par: {video.presenter}
          </p>
        )}
        
        <p className="text-gray-700 mb-4">{video.description}</p>
        
        <div className="flex flex-wrap gap-2 mb-4">
          {video.tags.map((tag) => (
            <span 
              key={`${video.id}-${tag}`}
              className="inline-flex items-center px-2 py-0.5 rounded-full text-xs font-medium bg-ecofor-light-blue/30 text-ecofor-dark-green"
            >
              <Tag className="w-3 h-3 mr-1" />
              {tag}
            </span>
          ))}
        </div>
        
        <a 
          href={`https://www.youtube.com/watch?v=${video.youtubeId}`}
          target="_blank"
          rel="noopener noreferrer"
          className="inline-flex items-center text-ecofor-dark-green hover:text-ecofor-turquoise transition-colors font-medium"
        >
          Voir sur YouTube
          <ExternalLink className="ml-1 h-4 w-4" />
        </a>
      </div>
    </div>
  );
};

export default VideoCard;
