
/**
 * Wrapper pour les pages WordPress
 * Ce composant permet de charger dynamiquement le contenu WordPress
 * et de l'afficher dans un composant React adapté
 */

import React, { useEffect, useState } from 'react';
import { useParams, useLocation } from 'react-router-dom';
import { getPageBySlug, wordpressConfig } from '@/wordpress-config';
import { getComponentForTemplate, convertRouteToTemplate } from '@/wordpress/TemplateMapping';
import EditorAdapter from '@/wordpress/EditorAdapter';
import PageLayout from '@/components/layout/PageLayout';
import { Skeleton } from '@/components/ui/skeleton';
import NotFound from '@/pages/NotFound';
import { toast } from '@/components/ui/use-toast';

interface WordPressPageProps {
  slug?: string;
  templateName?: string;
}

const WordPressPageWrapper: React.FC<WordPressPageProps> = ({ slug: propSlug, templateName }) => {
  const params = useParams();
  const location = useLocation();
  const slug = propSlug || params.slug || '';
  
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [pageData, setPageData] = useState<any>(null);
  
  // Déterminer le template à utiliser basé sur la route actuelle
  const currentTemplate = templateName || convertRouteToTemplate(location.pathname);
  
  useEffect(() => {
    const fetchPage = async () => {
      try {
        setLoading(true);
        
        // Si aucun slug n'est défini, chercher une page avec le template correspondant
        let fetchParams = slug ? { slug } : { template: currentTemplate };
        const data = await getPageBySlug(slug);
        
        if (!data || data.length === 0) {
          setError('Page non trouvée');
          toast({
            title: "Page non trouvée",
            description: `La page "${slug || currentTemplate}" n'existe pas dans WordPress`,
            variant: "destructive"
          });
          setLoading(false);
          return;
        }
        
        setPageData(data[0]);
        setError(null);
      } catch (err) {
        console.error('Erreur lors du chargement de la page:', err);
        setError('Une erreur est survenue lors du chargement de la page');
        toast({
          title: "Erreur de chargement",
          description: "Impossible de charger la page depuis WordPress",
          variant: "destructive"
        });
      } finally {
        setLoading(false);
      }
    };
    
    fetchPage();
  }, [slug, currentTemplate]);
  
  if (loading) {
    return (
      <PageLayout>
        <div className="container py-16">
          <Skeleton className="h-12 w-3/4 mb-6" />
          <Skeleton className="h-6 w-full mb-4" />
          <Skeleton className="h-6 w-full mb-4" />
          <Skeleton className="h-6 w-3/4 mb-8" />
          <Skeleton className="h-64 w-full mb-8" />
          <Skeleton className="h-6 w-full mb-4" />
          <Skeleton className="h-6 w-full mb-4" />
          <Skeleton className="h-6 w-2/3" />
        </div>
      </PageLayout>
    );
  }
  
  if (error || !pageData) {
    return <NotFound />;
  }
  
  // Déterminer le template WordPress utilisé
  const template = pageData.template || pageData.meta?.wp_template || currentTemplate || 'default';
  
  // Récupérer le composant React correspondant
  const PageComponent = getComponentForTemplate(template);
  
  // Si un composant spécifique est disponible, l'utiliser
  if (PageComponent && PageComponent !== NotFound) {
    return <PageComponent wordpressData={pageData} />;
  }
  
  // Sinon, utiliser l'adaptateur d'éditeur pour rendre le contenu WordPress
  return (
    <PageLayout>
      <div className="container py-16">
        <h1 className="mb-6 text-3xl font-bold text-forestt-primary">
          {pageData.title.rendered}
        </h1>
        
        {/* Afficher l'image à la une si elle existe */}
        {pageData.featured_media && pageData._embedded?.['wp:featuredmedia'] && (
          <div className="mb-8">
            <img 
              src={pageData._embedded['wp:featuredmedia'][0].source_url}
              alt={pageData._embedded['wp:featuredmedia'][0].alt_text || pageData.title.rendered}
              className="w-full h-auto rounded-lg max-w-3xl mx-auto"
            />
          </div>
        )}
        
        <EditorAdapter 
          content={pageData.content.rendered} 
          editor={pageData.meta?.editor_used || 'gutenberg'}
        />
      </div>
    </PageLayout>
  );
};

export default WordPressPageWrapper;
