
import React from 'react';
import { Link } from 'react-router-dom';
import { parseWordPressContent } from '../../wordpress-config';
import { formatDistanceToNow } from 'date-fns';
import { fr } from 'date-fns/locale';

interface WordPressPostProps {
  post: {
    id: number;
    title: { rendered: string };
    excerpt: { rendered: string };
    content: { rendered: string };
    date: string;
    slug: string;
    _embedded?: {
      'wp:featuredmedia'?: Array<{
        source_url: string;
        alt_text?: string;
      }>;
      'wp:term'?: Array<Array<{
        name: string;
        slug: string;
      }>>;
      author?: Array<{
        name: string;
      }>;
    };
  };
  isFeatured?: boolean;
  showExcerpt?: boolean;
  showFullContent?: boolean;
}

const WordPressPostRenderer: React.FC<WordPressPostProps> = ({ 
  post, 
  isFeatured = false,
  showExcerpt = true,
  showFullContent = false
}) => {
  if (!post) return null;

  // Extraire l'image à la une
  const featuredImage = post._embedded?.['wp:featuredmedia']?.[0]?.source_url;
  
  // Extraire les catégories
  const categories = post._embedded?.['wp:term']?.[0] || [];
  
  // Extraire l'auteur
  const author = post._embedded?.author?.[0]?.name || 'GIP Ecofor';
  
  // Formater la date
  const publishDate = new Date(post.date);
  const formattedDate = formatDistanceToNow(publishDate, { 
    addSuffix: true,
    locale: fr
  });

  // Nettoyer l'extrait ou le contenu du HTML
  const cleanExcerpt = post.excerpt?.rendered
    ? post.excerpt.rendered.replace(/<[^>]*>/g, '').substring(0, 150) + '...'
    : '';

  // Préparer le contenu si nécessaire
  const content = showFullContent 
    ? parseWordPressContent(post.content.rendered)
    : '';

  return (
    <article className={`wordpress-post ${isFeatured ? 'featured-post' : ''} mb-8 border rounded-lg overflow-hidden shadow-sm`}>
      {featuredImage && (
        <div className="post-featured-image">
          <img
            src={featuredImage}
            alt={`Image pour ${post.title.rendered}`}
            className="w-full h-48 object-cover"
          />
        </div>
      )}
      
      <div className="p-4">
        <div className="post-meta flex items-center text-sm text-gray-500 mb-2">
          <span className="mr-3">{formattedDate}</span>
          <span className="mr-3">Par {author}</span>
        </div>
        
        <h2 className="text-xl font-bold mb-2">
          {showFullContent ? (
            <span dangerouslySetInnerHTML={{ __html: post.title.rendered }} />
          ) : (
            <Link 
              to={`/news/${post.slug}`} 
              className="hover:text-ecofor-dark-green transition-colors"
            >
              <span dangerouslySetInnerHTML={{ __html: post.title.rendered }} />
            </Link>
          )}
        </h2>
        
        {categories.length > 0 && (
          <div className="post-categories flex flex-wrap gap-2 mb-3">
            {categories.map((category, index) => (
              <span 
                key={index} 
                className="text-xs bg-ecofor-light-blue text-ecofor-dark-green px-2 py-1 rounded-full"
              >
                {category.name}
              </span>
            ))}
          </div>
        )}
        
        {showExcerpt && !showFullContent && (
          <div className="post-excerpt mb-3">
            <p>{cleanExcerpt}</p>
          </div>
        )}
        
        {showFullContent && (
          <div 
            className="post-content mt-4"
            dangerouslySetInnerHTML={{ __html: content }}
          />
        )}
        
        {!showFullContent && (
          <div className="mt-4">
            <Link
              to={`/news/${post.slug}`}
              className="text-ecofor-dark-green hover:underline font-medium"
            >
              Lire la suite
            </Link>
          </div>
        )}
      </div>
    </article>
  );
};

export default WordPressPostRenderer;
