
import { useState, useEffect } from 'react';
import { fetchWordPressData, wordpressConfig } from '../wordpress-config';
import { toast } from '@/components/ui/use-toast';

interface UseWordPressContentParams {
  contentType: string;
  slug?: string;
  params?: Record<string, string>;
  cacheTime?: number;
}

interface UseWordPressContentReturn {
  content: any;
  loading: boolean;
  error: boolean;
  refetch: () => Promise<void>;
  meta: {
    title?: string;
    featuredImage?: string;
    author?: string;
    date?: Date;
    categories?: any[];
    tags?: any[];
  };
  acf?: Record<string, any>;
}

// Cache pour stocker les résultats des requêtes
const contentCache: Record<string, { data: any; timestamp: number }> = {};

const useWordPressContent = ({
  contentType,
  slug,
  params = {},
  cacheTime = wordpressConfig.cache.ttl
}: UseWordPressContentParams): UseWordPressContentReturn => {
  const [content, setContent] = useState<any>(null);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<boolean>(false);
  const [meta, setMeta] = useState<UseWordPressContentReturn['meta']>({});
  const [acf, setAcf] = useState<Record<string, any>>({});

  // Créer une clé de cache unique pour cette requête
  const cacheKey = `${contentType}:${slug || ''}:${JSON.stringify(params)}`;

  const fetchContent = async () => {
    setLoading(true);
    setError(false);
    
    // Vérifier le cache si activé
    if (wordpressConfig.cache.enabled) {
      const cachedData = contentCache[cacheKey];
      const now = Date.now();
      
      // Utiliser les données en cache si elles existent et sont toujours valides
      if (cachedData && (now - cachedData.timestamp) < cacheTime) {
        setContent(cachedData.data);
        extractMetadata(cachedData.data);
        setLoading(false);
        return;
      }
    }
    
    try {
      let data;
      if (slug) {
        data = await fetchWordPressData(contentType, { slug, ...params, _embed: true });
      } else {
        data = await fetchWordPressData(contentType, { ...params, _embed: true });
      }
      
      if (Array.isArray(data) && data.length > 0) {
        setContent(data[0]);
        extractMetadata(data[0]);
        
        // Mettre en cache les données
        if (wordpressConfig.cache.enabled) {
          contentCache[cacheKey] = { data: data[0], timestamp: Date.now() };
        }
      } else if (data && !Array.isArray(data)) {
        setContent(data);
        extractMetadata(data);
        
        // Mettre en cache les données
        if (wordpressConfig.cache.enabled) {
          contentCache[cacheKey] = { data, timestamp: Date.now() };
        }
      } else if (Array.isArray(data)) {
        setContent(data);
        
        // Mettre en cache les données
        if (wordpressConfig.cache.enabled) {
          contentCache[cacheKey] = { data, timestamp: Date.now() };
        }
      } else {
        setError(true);
        toast({
          title: "Contenu non trouvé",
          description: "Impossible de charger le contenu depuis WordPress",
          variant: "destructive"
        });
      }
    } catch (err) {
      console.error("Erreur lors du chargement du contenu WordPress:", err);
      setError(true);
      toast({
        title: "Erreur de chargement",
        description: "Problème de connexion à l'API WordPress",
        variant: "destructive"
      });
    } finally {
      setLoading(false);
    }
  };
  
  // Fonction pour extraire les métadonnées du contenu WordPress
  const extractMetadata = (data: any) => {
    if (!data) return;
    
    // Extraire les informations de base
    const newMeta: UseWordPressContentReturn['meta'] = {
      title: data.title?.rendered || data.title,
      featuredImage: data.featured_media && data._embedded?.['wp:featuredmedia']?.[0]?.source_url,
      author: data._embedded?.['author']?.[0]?.name,
      date: data.date ? new Date(data.date) : undefined,
      categories: data._embedded?.['wp:term']?.[0] || [],
      tags: data._embedded?.['wp:term']?.[1] || [],
    };
    
    setMeta(newMeta);
    
    // Extraire les champs ACF
    if (data.acf) {
      setAcf(data.acf);
    }
  };

  useEffect(() => {
    fetchContent();
  }, [contentType, slug, JSON.stringify(params)]);

  // Nettoyer le cache si nécessaire
  const invalidateCache = () => {
    if (contentCache[cacheKey]) {
      delete contentCache[cacheKey];
    }
  };

  return { 
    content, 
    loading, 
    error, 
    refetch: fetchContent,
    meta,
    acf
  };
};

// Fonction utilitaire pour invalider tout le cache
export const invalidateAllCache = () => {
  Object.keys(contentCache).forEach(key => {
    delete contentCache[key];
  });
};

export default useWordPressContent;
