
import { Helmet } from "react-helmet-async";
import PageLayout from "@/components/layout/PageLayout";
import { useState, useEffect } from "react";
import { Calendar, MapPin, Tag, ChevronRight, Search, Filter } from "lucide-react";
import { Button } from "@/components/ui/button";
import NewsCard from "@/components/news/NewsCard";
import NewsSearch from "@/components/news/NewsSearch";
import { newsItems } from "@/data/newsItems";
import {
  Pagination,
  PaginationContent,
  PaginationEllipsis,
  PaginationItem,
  PaginationLink,
  PaginationNext,
  PaginationPrevious,
} from "@/components/ui/pagination";

const ITEMS_PER_PAGE = 6;

const News = () => {
  const [searchTerm, setSearchTerm] = useState("");
  const [activeTab, setActiveTab] = useState("all");
  const [currentPage, setCurrentPage] = useState(1);
  const [filteredItems, setFilteredItems] = useState(newsItems);
  
  // Filtrer les éléments en fonction de la recherche et de l'onglet actif
  useEffect(() => {
    let results = newsItems;
    
    // Filtrer par type (onglet)
    if (activeTab !== "all") {
      results = results.filter(item => item.type === activeTab);
    }
    
    // Filtrer par terme de recherche
    if (searchTerm.trim() !== "") {
      const searchLower = searchTerm.toLowerCase();
      results = results.filter(item => 
        item.title.toLowerCase().includes(searchLower) || 
        item.excerpt.toLowerCase().includes(searchLower) ||
        (item.tags && item.tags.some(tag => tag.toLowerCase().includes(searchLower)))
      );
    }
    
    setFilteredItems(results);
    setCurrentPage(1); // Réinitialiser à la première page après un filtrage
  }, [searchTerm, activeTab]);
  
  // Calcul de la pagination
  const totalPages = Math.ceil(filteredItems.length / ITEMS_PER_PAGE);
  const startIndex = (currentPage - 1) * ITEMS_PER_PAGE;
  const paginatedItems = filteredItems.slice(startIndex, startIndex + ITEMS_PER_PAGE);
  
  // Tabs for filtering
  const tabs = [
    { id: "all", label: "Tous" },
    { id: "news", label: "Actualités" },
    { id: "events", label: "Événements" }
  ];
  
  // Date formatée pour les événements à venir
  const formatEventDate = (dateString: string) => {
    const date = new Date(dateString);
    const day = date.getDate();
    const month = date.toLocaleDateString('fr-FR', { month: 'long' });
    const year = date.getFullYear();
    return { day, month, year };
  };
  
  // Événements à venir (triés par date)
  const upcomingEvents = newsItems
    .filter(item => item.type === "events")
    .sort((a, b) => new Date(a.date).getTime() - new Date(b.date).getTime())
    .slice(0, 3);
  
  return (
    <PageLayout>
      <Helmet>
        <title>Actualités & Événements - FORESTT-HUB</title>
        <meta name="description" content="Dernières actualités et événements du projet FORESTT-HUB concernant la recherche sur les forêts." />
      </Helmet>
      
      <section className="py-12 md:py-16">
        <div className="container">
          <div className="max-w-3xl mx-auto text-center mb-10">
            <h1 className="text-3xl md:text-4xl font-bold mb-4 text-ecofor-dark-green">Actualités & Événements</h1>
            <p className="text-lg text-gray-600">
              Retrouvez les dernières nouvelles et événements liés au projet FORESTT-HUB.
            </p>
          </div>
          
          {/* Recherche et filtres */}
          <NewsSearch 
            searchTerm={searchTerm}
            setSearchTerm={setSearchTerm}
            activeTab={activeTab}
            setActiveTab={setActiveTab}
            tabs={tabs}
          />
          
          {/* Résultats */}
          <div className="mt-10">
            {filteredItems.length === 0 ? (
              <div className="text-center py-12">
                <p className="text-lg text-gray-500">Aucun résultat trouvé pour votre recherche.</p>
                <Button 
                  variant="outline" 
                  className="mt-4"
                  onClick={() => {
                    setSearchTerm("");
                    setActiveTab("all");
                  }}
                >
                  Réinitialiser les filtres
                </Button>
              </div>
            ) : (
              <>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                  {paginatedItems.map((item, index) => (
                    <NewsCard key={item.id} item={item} index={index} />
                  ))}
                </div>
                
                {/* Pagination */}
                {totalPages > 1 && (
                  <Pagination className="my-8">
                    <PaginationContent>
                      {currentPage > 1 && (
                        <PaginationItem>
                          <PaginationPrevious onClick={() => setCurrentPage(currentPage - 1)} />
                        </PaginationItem>
                      )}
                      
                      {Array.from({ length: totalPages }).map((_, i) => {
                        // Afficher les premières pages, les dernières et les pages autour de la page courante
                        if (
                          i === 0 || 
                          i === totalPages - 1 || 
                          (i >= currentPage - 2 && i <= currentPage + 1 && i < totalPages - 1)
                        ) {
                          return (
                            <PaginationItem key={i}>
                              <PaginationLink 
                                isActive={currentPage === i + 1}
                                onClick={() => setCurrentPage(i + 1)}
                              >
                                {i + 1}
                              </PaginationLink>
                            </PaginationItem>
                          );
                        } else if (i === currentPage + 2 && i < totalPages - 2) {
                          return <PaginationItem key="ellipsis"><PaginationEllipsis /></PaginationItem>;
                        }
                        return null;
                      })}
                      
                      {currentPage < totalPages && (
                        <PaginationItem>
                          <PaginationNext onClick={() => setCurrentPage(currentPage + 1)} />
                        </PaginationItem>
                      )}
                    </PaginationContent>
                  </Pagination>
                )}
              </>
            )}
          </div>
        </div>
      </section>
      
      {/* Section des événements à venir */}
      <section className="py-12 bg-ecofor-light-blue/10">
        <div className="container">
          <h2 className="text-2xl font-bold mb-8 text-center">Événements à venir</h2>
          <div className="max-w-4xl mx-auto">
            <div className="space-y-6">
              {upcomingEvents.map((event) => {
                const { day, month, year } = formatEventDate(event.date);
                return (
                  <div key={event.id} className="bg-white p-5 rounded-lg shadow-sm border border-ecofor-medium-blue/20 flex flex-col md:flex-row gap-6">
                    <div className="md:w-1/4 flex-shrink-0">
                      <div className="bg-ecofor-dark-green/10 text-ecofor-dark-green rounded-lg p-4 text-center">
                        <span className="block text-2xl font-bold">{day}</span>
                        <span className="block text-sm uppercase">{month}</span>
                        <span className="block text-sm">{year}</span>
                      </div>
                    </div>
                    <div className="md:w-3/4">
                      <h3 className="text-xl font-semibold text-ecofor-dark-green hover:text-ecofor-turquoise transition-colors">
                        <a href={`/news/${event.slug}`}>{event.title}</a>
                      </h3>
                      
                      {event.location && (
                        <p className="text-sm text-gray-600 mt-1 flex items-center">
                          <MapPin className="w-4 h-4 mr-1" />
                          {event.location}
                        </p>
                      )}
                      
                      <p className="mt-3 text-gray-700">{event.excerpt}</p>
                      
                      <div className="mt-4 flex flex-wrap items-center justify-between">
                        <div className="flex flex-wrap gap-2 mb-2 md:mb-0">
                          {event.tags && event.tags.map((tag) => (
                            <span 
                              key={`${event.id}-${tag}`}
                              className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-ecofor-light-blue/30 text-ecofor-dark-green"
                            >
                              <Tag className="w-3 h-3 mr-1" />
                              {tag}
                            </span>
                          ))}
                        </div>
                        
                        <a 
                          href={`/news/${event.slug}`}
                          className="inline-flex items-center text-ecofor-turquoise hover:text-ecofor-dark-green font-medium text-sm transition-colors"
                        >
                          Plus d'informations
                          <ChevronRight className="w-4 h-4 ml-1" />
                        </a>
                      </div>
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        </div>
      </section>
      
      {/* Bannière d'inscription à la newsletter */}
      <section className="py-12 bg-ecofor-dark-green text-white">
        <div className="container">
          <div className="max-w-4xl mx-auto text-center">
            <h2 className="text-2xl font-bold mb-4">Recevez nos actualités directement dans votre boîte mail</h2>
            <p className="mb-6">Pour suivre les actualités du projet-ciblé FORESTT-HUB et du PEPR FORESTT dans son ensemble, vous pouvez vous abonner à la newsletter du programme.</p>
            
            <div className="flex justify-center">
              <a 
                href="https://www.pepr-forestt.org/news-notification-user" 
                target="_blank" 
                rel="noopener noreferrer"
                className="inline-block"
              >
                <Button className="bg-ecofor-turquoise hover:bg-ecofor-turquoise/90 text-white font-medium">
                  S'inscrire à la newsletter
                </Button>
              </a>
            </div>
          </div>
        </div>
      </section>
    </PageLayout>
  );
};

export default News;
