
import { useParams, Link } from "react-router-dom";
import { useState, useEffect } from "react";
import { Helmet } from "react-helmet-async";
import PageLayout from "@/components/layout/PageLayout";
import { ChevronLeft, Calendar, MapPin, Tag, Share2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Skeleton } from "@/components/ui/skeleton";
import { newsItems } from "@/data/newsItems";

const NewsDetail = () => {
  const { slug } = useParams<{ slug: string }>();
  const [newsItem, setNewsItem] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [relatedItems, setRelatedItems] = useState<any[]>([]);

  useEffect(() => {
    // Simuler un chargement d'API
    setLoading(true);
    setTimeout(() => {
      // Trouver l'élément correspondant au slug
      const item = newsItems.find(item => item.slug === slug);
      setNewsItem(item || null);
      
      // Trouver des éléments similaires (même type ou tags similaires)
      if (item) {
        const related = newsItems
          .filter(i => i.id !== item.id)
          .filter(i => i.type === item.type || (
            item.tags && i.tags && i.tags.some(tag => item.tags.includes(tag))
          ))
          .slice(0, 3);
        setRelatedItems(related);
      }
      
      setLoading(false);
    }, 500);
  }, [slug]);

  // Formater la date
  const formatDate = (dateString: string) => {
    const options: Intl.DateTimeFormatOptions = { 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    };
    return new Date(dateString).toLocaleDateString('fr-FR', options);
  };

  if (loading) {
    return (
      <PageLayout>
        <div className="container py-12">
          <div className="max-w-4xl mx-auto">
            <Skeleton className="h-8 w-2/3 mb-4" />
            <Skeleton className="h-6 w-48 mb-8" />
            <Skeleton className="h-72 w-full mb-8" />
            <Skeleton className="h-4 w-full mb-2" />
            <Skeleton className="h-4 w-full mb-2" />
            <Skeleton className="h-4 w-full mb-2" />
            <Skeleton className="h-4 w-3/4 mb-8" />
            <Skeleton className="h-4 w-full mb-2" />
            <Skeleton className="h-4 w-full mb-2" />
            <Skeleton className="h-4 w-2/3 mb-4" />
          </div>
        </div>
      </PageLayout>
    );
  }

  if (!newsItem) {
    return (
      <PageLayout>
        <div className="container py-12">
          <div className="max-w-4xl mx-auto text-center">
            <h1 className="text-2xl font-bold mb-4 text-ecofor-dark-green">Actualité non trouvée</h1>
            <p className="mb-6">L'actualité ou l'événement que vous recherchez n'existe pas ou a été supprimé.</p>
            <Link to="/news">
              <Button>Retour aux actualités</Button>
            </Link>
          </div>
        </div>
      </PageLayout>
    );
  }

  return (
    <PageLayout>
      <Helmet>
        <title>{newsItem.title} - FORESTT-HUB</title>
        <meta name="description" content={newsItem.excerpt} />
      </Helmet>
      
      <article className="py-12">
        <div className="container">
          <div className="max-w-4xl mx-auto">
            {/* Navigation retour */}
            <Link to="/news" className="inline-flex items-center text-ecofor-dark-green hover:text-ecofor-turquoise mb-8">
              <ChevronLeft className="w-5 h-5 mr-1" />
              Retour aux actualités
            </Link>
            
            {/* En-tête */}
            <h1 className="text-3xl md:text-4xl font-bold mb-4 text-ecofor-dark-green">
              {newsItem.title}
            </h1>
            
            <div className="flex flex-wrap items-center gap-4 mb-8 text-gray-600">
              <div className="flex items-center">
                <Calendar className="w-4 h-4 mr-1" />
                <time dateTime={newsItem.date}>{formatDate(newsItem.date)}</time>
              </div>
              
              {newsItem.location && (
                <div className="flex items-center">
                  <MapPin className="w-4 h-4 mr-1" />
                  <span>{newsItem.location}</span>
                </div>
              )}
              
              {newsItem.type === "events" && (
                <span className="bg-ecofor-turquoise text-white px-3 py-1 rounded-full text-sm">
                  Événement
                </span>
              )}
              
              {newsItem.type === "news" && (
                <span className="bg-ecofor-light-blue text-ecofor-dark-green px-3 py-1 rounded-full text-sm">
                  Actualité
                </span>
              )}
            </div>
            
            {/* Image principale */}
            <div className="mb-8 rounded-xl overflow-hidden">
              <img 
                src={newsItem.image} 
                alt={newsItem.title} 
                className="w-full h-auto"
              />
            </div>
            
            {/* Contenu */}
            <div className="prose prose-lg max-w-none mb-8">
              {/* Contenu d'exemple */}
              <p>
                Le GIP Ecofor, à travers son projet FORESTT-HUB, a le plaisir de vous présenter cette {newsItem.type === "events" ? "événement" : "actualité"} qui s'inscrit dans notre mission de recherche et développement sur les forêts françaises et européennes.
              </p>
              
              <p>
                Les écosystèmes forestiers font face à de nombreux défis liés au changement climatique, à la préservation de la biodiversité et à la gestion durable des ressources. À travers ce projet, nous visons à développer des solutions innovantes et collaboratives.
              </p>
              
              <h2>Objectifs et perspectives</h2>
              
              <p>
                Cette initiative permet de rassembler chercheurs, gestionnaires forestiers et décideurs politiques autour d'enjeux communs. Elle s'inscrit dans une démarche collaborative essentiellement pour le partage des connaissances et le développement d'outils d'aide à la décision.
              </p>
              
              {newsItem.type === "events" && (
                <>
                  <h2>Informations pratiques</h2>
                  <ul>
                    <li><strong>Date :</strong> {formatDate(newsItem.date)}</li>
                    <li><strong>Lieu :</strong> {newsItem.location || "À préciser"}</li>
                    <li><strong>Inscription :</strong> Obligatoire, via le formulaire en ligne</li>
                  </ul>
                  
                  <div className="not-prose mt-8">
                    <Button className="bg-ecofor-turquoise hover:bg-ecofor-dark-green text-white">
                      S'inscrire à cet événement
                    </Button>
                  </div>
                </>
              )}
            </div>
            
            {/* Tags */}
            {newsItem.tags && newsItem.tags.length > 0 && (
              <div className="border-t border-gray-200 pt-6 mt-8">
                <h3 className="text-lg font-semibold mb-3">Tags associés</h3>
                <div className="flex flex-wrap gap-2">
                  {newsItem.tags.map((tag: string) => (
                    <span 
                      key={tag}
                      className="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-ecofor-light-blue/30 text-ecofor-dark-green"
                    >
                      <Tag className="w-3 h-3 mr-1" />
                      {tag}
                    </span>
                  ))}
                </div>
              </div>
            )}
            
            {/* Partage */}
            <div className="border-t border-gray-200 pt-6 mt-8">
              <h3 className="text-lg font-semibold mb-3">Partager</h3>
              <div className="flex gap-2">
                <Button variant="outline" size="sm" className="flex gap-2 items-center">
                  <Share2 className="w-4 h-4" />
                  LinkedIn
                </Button>
                <Button variant="outline" size="sm" className="flex gap-2 items-center">
                  <Share2 className="w-4 h-4" />
                  Twitter
                </Button>
                <Button variant="outline" size="sm" className="flex gap-2 items-center">
                  <Share2 className="w-4 h-4" />
                  Email
                </Button>
              </div>
            </div>
          </div>
        </div>
      </article>
      
      {/* Articles similaires */}
      {relatedItems.length > 0 && (
        <section className="py-12 bg-ecofor-light-blue/10">
          <div className="container">
            <h2 className="text-2xl font-bold mb-8 text-center">Articles similaires</h2>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-8 max-w-4xl mx-auto">
              {relatedItems.map((item, index) => (
                <div 
                  key={item.id}
                  className="bg-white rounded-lg shadow-sm overflow-hidden border border-ecofor-medium-blue/20 hover:shadow-md transition-shadow"
                >
                  <Link to={`/news/${item.slug}`}>
                    <div className="h-40 overflow-hidden">
                      <img 
                        src={item.image} 
                        alt={item.title}
                        className="w-full h-full object-cover hover:scale-105 transition-transform duration-300"
                      />
                    </div>
                    <div className="p-4">
                      <span className="text-sm text-gray-500 mb-1 block">
                        {formatDate(item.date)}
                      </span>
                      <h3 className="font-semibold text-ecofor-dark-green hover:text-ecofor-turquoise transition-colors">
                        {item.title}
                      </h3>
                    </div>
                  </Link>
                </div>
              ))}
            </div>
          </div>
        </section>
      )}
    </PageLayout>
  );
};

export default NewsDetail;
