
import { Helmet } from "react-helmet-async";
import PageLayout from "@/components/layout/PageLayout";
import { useState } from "react";
import ResourceTabs from "@/components/resources/ResourceTabs";
import { 
  Pagination,
  PaginationContent,
  PaginationEllipsis,
  PaginationItem,
  PaginationLink,
  PaginationNext,
  PaginationPrevious,
} from "@/components/ui/pagination";
import PublicationCard from "@/components/resources/PublicationCard";
import FilterTags from "@/components/resources/FilterTags";
import { publicationsData } from "@/data/publications";

const Publications = () => {
  const [currentPage, setCurrentPage] = useState(1);
  const [activeFilter, setActiveFilter] = useState<string | null>(null);
  
  const itemsPerPage = 4;
  const categories = Array.from(new Set(publicationsData.flatMap(pub => pub.tags)));
  
  // Créer le tableau de filtres au format attendu par FilterTags
  const filters = categories.map(category => ({
    id: category,
    label: category
  }));
  
  // Filtrer les publications en fonction du filtre actif
  const filteredPublications = activeFilter === null 
    ? publicationsData 
    : publicationsData.filter(pub => pub.tags.includes(activeFilter));
    
  const pageCount = Math.ceil(filteredPublications.length / itemsPerPage);
  const startIndex = (currentPage - 1) * itemsPerPage;
  const currentItems = filteredPublications.slice(startIndex, startIndex + itemsPerPage);
  
  const handleFilterChange = (filterId: string) => {
    setActiveFilter(filterId);
    setCurrentPage(1); // Réinitialiser la pagination lors du changement de filtre
  };
  
  const clearFilters = () => {
    setActiveFilter(null);
    setCurrentPage(1);
  };

  return (
    <PageLayout>
      <Helmet>
        <title>Publications - FORESTT-HUB</title>
        <meta name="description" content="Publications scientifiques du projet FORESTT-HUB concernant la recherche forestière." />
      </Helmet>
      
      <section className="py-12 md:py-16">
        <div className="container">
          <div className="max-w-3xl mx-auto text-center mb-10">
            <h1 className="text-3xl md:text-4xl font-bold mb-4 text-ecofor-dark-green">Publications scientifiques</h1>
            <p className="text-lg text-gray-600">
              Découvrez nos publications scientifiques récentes sur la recherche forestière et les écosystèmes.
            </p>
          </div>
          
          <ResourceTabs />
          
          <FilterTags 
            filters={filters} 
            activeFilter={activeFilter} 
            onChange={handleFilterChange} 
            onClear={clearFilters}
          />
          
          <div className="space-y-6 mb-10">
            {currentItems.map((publication) => (
              <PublicationCard key={publication.id} publication={publication} />
            ))}
          </div>
          
          {pageCount > 1 && (
            <Pagination className="my-8">
              <PaginationContent>
                {currentPage > 1 && (
                  <PaginationItem>
                    <PaginationPrevious onClick={() => setCurrentPage(currentPage - 1)} />
                  </PaginationItem>
                )}
                
                {Array.from({ length: pageCount }).map((_, i) => {
                  if (i === 0 || i === pageCount - 1 || (i >= currentPage - 2 && i <= currentPage)) {
                    return (
                      <PaginationItem key={i}>
                        <PaginationLink 
                          isActive={currentPage === i + 1}
                          onClick={() => setCurrentPage(i + 1)}
                        >
                          {i + 1}
                        </PaginationLink>
                      </PaginationItem>
                    );
                  } else if (i === currentPage + 1 && currentPage < pageCount - 2) {
                    return (
                      <PaginationItem key="ellipsis">
                        <PaginationEllipsis />
                      </PaginationItem>
                    );
                  }
                  return null;
                })}
                
                {currentPage < pageCount && (
                  <PaginationItem>
                    <PaginationNext onClick={() => setCurrentPage(currentPage + 1)} />
                  </PaginationItem>
                )}
              </PaginationContent>
            </Pagination>
          )}
        </div>
      </section>
    </PageLayout>
  );
};

export default Publications;
