
import { Helmet } from "react-helmet-async";
import PageLayout from "@/components/layout/PageLayout";
import { useState } from "react";
import ResourceTabs from "@/components/resources/ResourceTabs";
import FilterTags from "@/components/resources/FilterTags";
import VideoCard from "@/components/resources/VideoCard";
import EmptyState from "@/components/resources/EmptyState";
import { videosData } from "@/data/videos";

const Videos = () => {
  const [activeFilter, setActiveFilter] = useState<string | null>(null);
  
  const allTags = Array.from(new Set(videosData.flatMap(video => video.tags)));
  
  // Créer le tableau de filtres au format attendu par FilterTags
  const filters = allTags.map(tag => ({
    id: tag,
    label: tag
  }));
  
  const filteredVideos = activeFilter === null 
    ? videosData 
    : videosData.filter(video => video.tags.includes(activeFilter));

  const handleFilterChange = (filterId: string) => {
    setActiveFilter(filterId);
  };
  
  const clearFilters = () => {
    setActiveFilter(null);
  };

  return (
    <PageLayout>
      <Helmet>
        <title>Vidéos - FORESTT-HUB</title>
        <meta name="description" content="Vidéos, webinaires et présentations du projet FORESTT-HUB sur la recherche forestière." />
      </Helmet>
      
      <section className="py-12 md:py-16">
        <div className="container">
          <div className="max-w-3xl mx-auto text-center mb-10">
            <h1 className="text-3xl md:text-4xl font-bold mb-4 text-ecofor-dark-green">Vidéos</h1>
            <p className="text-lg text-gray-600">
              Découvrez nos présentations, webinaires et tutoriels sur la recherche forestière et le projet FORESTT-HUB.
            </p>
          </div>
          
          <ResourceTabs />
          
          <FilterTags 
            filters={filters} 
            activeFilter={activeFilter} 
            onChange={handleFilterChange} 
            onClear={clearFilters}
          />
          
          <div className="grid gap-8 sm:grid-cols-1 lg:grid-cols-2">
            {filteredVideos.map((video) => (
              <VideoCard key={video.id} video={video} />
            ))}
          </div>
          
          {filteredVideos.length === 0 && (
            <EmptyState 
              message="Aucune vidéo trouvée pour cette catégorie." 
              buttonText="Voir toutes les vidéos"
              onButtonClick={clearFilters}
            />
          )}
        </div>
      </section>
    </PageLayout>
  );
};

export default Videos;
