/**
 * Configuration pour l'intégration WordPress
 * Ce fichier permet de définir les points d'accès pour l'API WordPress
 * et de configurer les différents blocs et éditeurs supportés.
 */

// Configuration de base
export const wordpressConfig = {
  // URL de l'API WordPress - Mettre à jour avec l'URL de votre instance WordPress
  apiUrl: 'https://forestt-hub.gip-ecofor.org/wp-json/wp/v2',
  
  // Types de contenu à utiliser
  contentTypes: {
    pages: 'pages',
    posts: 'posts',
    customPost: 'project_materials',
    media: 'media',
    events: 'evenements',
    downloads: 'telecharger',
    partners: 'partenaires',
    workPackages: 'work_packages',
    publications: 'publications', 
    presentations: 'presentations',
    videos: 'videos',
    data: 'data_resources'
  },
  
  // Configuration des éditeurs supportés
  supportedEditors: {
    gutenberg: true,
    elementor: true, 
    wpbakery: true,
    divi: true
  },
  
  // Configuration des taxonomies
  taxonomies: {
    categories: 'categories',
    tags: 'tags',
    resourceTypes: 'resource_types',
    partnerTypes: 'partner_types',
    wpCategories: 'wp_categories'
  },
  
  // Configuration des blocs personnalisés
  customBlocks: {
    enabled: true,
    namespace: 'forestt-hub'
  },
  
  // Templates WordPress correspondants aux routes React
  templates: {
    'about': 'template-about',
    'about/presentation': 'template-presentation',
    'about/work-packages': 'template-work-packages',
    'about/partners': 'template-partners',
    'about/collaborators': 'template-collaborators',
    'resources': 'template-resources',
    'resources/deliverables': 'template-deliverables',
    'resources/publications': 'template-publications',
    'resources/presentations': 'template-presentations',
    'resources/data': 'template-data',
    'resources/videos': 'template-videos',
    'activites': 'template-activites',
    'activites/doctoral-schools': 'template-doctoral-schools',
    'activites/colloques': 'template-colloques',
    'activites/residences': 'template-residences',
    'activites/living-labs': 'template-living-labs',
    'activites/webinaires': 'template-webinaires',
    'activites/reseaux-formation': 'template-reseaux-formation',
    'news': 'template-news',
    'contact': 'template-contact'
  },
  
  // Paramètres d'authentification
  auth: {
    useJWT: true,
    tokenStorageKey: 'forestt_hub_auth_token'
  },
  
  // Configuration du cache
  cache: {
    enabled: true,
    ttl: 5 * 60 * 1000, // 5 minutes
    invalidateOnEdit: true
  }
};

// Utilitaires pour récupérer les données WordPress
export const fetchWordPressData = async (endpoint: string, params: Record<string, any> = {}) => {
  const queryString = new URLSearchParams(params).toString();
  const url = `${wordpressConfig.apiUrl}/${endpoint}?${queryString}`;
  
  try {
    const response = await fetch(url);
    if (!response.ok) {
      throw new Error(`WordPress API error: ${response.status}`);
    }
    return await response.json();
  } catch (error) {
    console.error('Erreur lors de la récupération des données WordPress:', error);
    return null;
  }
};

// Fonction pour récupérer une page par son slug
export const getPageBySlug = async (slug: string) => {
  return fetchWordPressData(wordpressConfig.contentTypes.pages, {
    slug,
    _embed: true
  });
};

// Fonction pour récupérer un article par son slug
export const getPostBySlug = async (slug: string) => {
  return fetchWordPressData(wordpressConfig.contentTypes.posts, {
    slug,
    _embed: true
  });
};

// Fonction pour récupérer les derniers articles
export const getLatestPosts = async (count = 3) => {
  return fetchWordPressData(wordpressConfig.contentTypes.posts, {
    per_page: count,
    _embed: true
  });
};

// Fonction pour récupérer les éléments de projet
export const getProjectMaterials = async (count = 10) => {
  return fetchWordPressData(wordpressConfig.contentTypes.customPost, {
    per_page: count,
    _embed: true
  });
};

// Fonction pour récupérer les médias
export const getMediaById = async (id: number | string) => {
  return fetchWordPressData(`${wordpressConfig.contentTypes.media}/${id}`);
};

// Fonction pour récupérer les événements
export const getEvents = async (count = 10) => {
  return fetchWordPressData(wordpressConfig.contentTypes.events, {
    per_page: count,
    _embed: true
  });
};

// Fonction pour récupérer les téléchargements
export const getDownloads = async (category = '', count = 20) => {
  const params: Record<string, any> = { per_page: count, _embed: true };
  if (category) {
    params.category = category;
  }
  return fetchWordPressData(wordpressConfig.contentTypes.downloads, params);
};

// Fonction pour récupérer les partenaires
export const getPartners = async (type = '') => {
  const params: Record<string, any> = { per_page: 100, _embed: true };
  if (type) {
    params[wordpressConfig.taxonomies.partnerTypes] = type;
  }
  return fetchWordPressData(wordpressConfig.contentTypes.partners, params);
};

// Fonction pour récupérer les work packages
export const getWorkPackages = async () => {
  return fetchWordPressData(wordpressConfig.contentTypes.workPackages, {
    per_page: 100,
    _embed: true
  });
};

// Fonction pour récupérer les publications
export const getPublications = async (params: Record<string, any> = {}) => {
  return fetchWordPressData(wordpressConfig.contentTypes.publications, {
    per_page: 20,
    _embed: true,
    ...params
  });
};

// Fonction pour récupérer les présentations
export const getPresentations = async (params: Record<string, any> = {}) => {
  return fetchWordPressData(wordpressConfig.contentTypes.presentations, {
    per_page: 20,
    _embed: true,
    ...params
  });
};

// Fonction pour récupérer les vidéos
export const getVideos = async (params: Record<string, any> = {}) => {
  return fetchWordPressData(wordpressConfig.contentTypes.videos, {
    per_page: 20,
    _embed: true,
    ...params
  });
};

// Fonction pour récupérer les données
export const getDataResources = async (params: Record<string, any> = {}) => {
  return fetchWordPressData(wordpressConfig.contentTypes.data, {
    per_page: 20,
    _embed: true,
    ...params
  });
};

// Utilitaire pour parser le contenu WordPress
export const parseWordPressContent = (content: string) => {
  if (!content) return '';
  
  // Remplacement des shortcodes WordPress par des composants React
  let parsedContent = content
    .replace(/\[contact-form-7(.*?)\]/g, '<ContactForm7 $1 />')
    .replace(/\[elementor-template(.*?)\]/g, '<ElementorTemplate $1 />')
    .replace(/\[gallery(.*?)\]/g, '<Gallery $1 />');
  
  // Traitement spécifique pour les vidéos YouTube
  parsedContent = parsedContent.replace(
    /<iframe.*?src=".*?youtube\.com\/embed\/(.*?)".*?><\/iframe>/g,
    '<div class="youtube-embed-container" data-video-id="$1"></div>'
  );
    
  return parsedContent;
};

// Mapper les blocs Gutenberg aux composants React
export const blockMapper: Record<string, any> = {
  'core/paragraph': 'p',
  'core/heading': ({level, content}: {level: number, content: string}) => `h${level}`,
  'core/image': 'img',
  'core/gallery': 'Gallery',
  'core/list': 'ul',
  'core/quote': 'blockquote',
  'core/button': 'Button',
  'core/columns': 'div',
  'core/embed': 'YouTubeEmbed',
  'core/file': 'DownloadButton',
  'core/table': 'Table'
};

// Fonction pour créer un nouveau contenu WordPress
export const createWordPressContent = async (contentType: string, data: any, token: string) => {
  try {
    const response = await fetch(`${wordpressConfig.apiUrl}/${contentType}`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify(data)
    });
    
    if (!response.ok) {
      throw new Error(`WordPress API error: ${response.status}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('Erreur lors de la création de contenu WordPress:', error);
    return null;
  }
};

// Fonctions d'authentification
export const authenticateWordPress = async (username: string, password: string) => {
  try {
    const response = await fetch(`${wordpressConfig.apiUrl.replace('/wp/v2', '')}/jwt-auth/v1/token`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ username, password })
    });
    
    if (!response.ok) {
      throw new Error("Erreur d'authentification");
    }
    
    const data = await response.json();
    
    // Stocker le token JWT
    if (data.token) {
      localStorage.setItem(wordpressConfig.auth.tokenStorageKey, data.token);
      return true;
    }
    
    return false;
  } catch (error) {
    console.error('Erreur d\'authentification WordPress:', error);
    return false;
  }
};

// Vérifier si l'utilisateur est authentifié
export const isAuthenticated = () => {
  const token = localStorage.getItem(wordpressConfig.auth.tokenStorageKey);
  return !!token;
};

// Récupérer le token d'authentification
export const getAuthToken = () => {
  return localStorage.getItem(wordpressConfig.auth.tokenStorageKey);
};

// Déconnexion
export const logoutWordPress = () => {
  localStorage.removeItem(wordpressConfig.auth.tokenStorageKey);
};
