
<?php
/**
 * FORESTT-HUB WordPress Theme functions
 *
 * @package FORESTT-HUB
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Définir les constantes du thème
define('FORESTT_THEME_VERSION', '1.0.0');
define('FORESTT_THEME_DIR', get_template_directory());
define('FORESTT_THEME_URI', get_template_directory_uri());

/**
 * Configuration du thème
 */
function forestt_theme_setup() {
    // Support des fonctionnalités WordPress
    add_theme_support('title-tag');
    add_theme_support('post-thumbnails');
    add_theme_support('html5', array(
        'search-form', 'comment-form', 'comment-list', 'gallery', 'caption'
    ));
    add_theme_support('customize-selective-refresh-widgets');
    add_theme_support('wp-block-styles');
    add_theme_support('responsive-embeds');

    // Enregistrement des menus
    register_nav_menus(array(
        'primary-menu' => esc_html__('Menu Principal', 'forestt-hub'),
        'footer-menu' => esc_html__('Menu Pied de Page', 'forestt-hub'),
    ));
}
add_action('after_setup_theme', 'forestt_theme_setup');

/**
 * Enregistrement des scripts et styles
 */
function forestt_enqueue_scripts() {
    // Styles principaux
    wp_enqueue_style(
        'forestt-styles',
        FORESTT_THEME_URI . '/style.css',
        array(),
        FORESTT_THEME_VERSION
    );

    // Scripts principaux
    wp_enqueue_script(
        'forestt-scripts',
        FORESTT_THEME_URI . '/js/main.js',
        array('jquery'),
        FORESTT_THEME_VERSION,
        true
    );

    // Variables JS pour l'API
    wp_localize_script('forestt-scripts', 'foresttApiSettings', array(
        'root' => esc_url_raw(rest_url()),
        'nonce' => wp_create_nonce('wp_rest'),
    ));
}
add_action('wp_enqueue_scripts', 'forestt_enqueue_scripts');

/**
 * Enregistrement des types de contenu personnalisés
 */
function forestt_register_post_types() {
    // Work Packages
    register_post_type('work_packages', array(
        'labels' => array(
            'name' => __('Work Packages', 'forestt-hub'),
            'singular_name' => __('Work Package', 'forestt-hub'),
        ),
        'public' => true,
        'has_archive' => true,
        'supports' => array('title', 'editor', 'excerpt', 'thumbnail'),
        'menu_icon' => 'dashicons-clipboard',
        'show_in_rest' => true,
    ));

    // Publications
    register_post_type('publications', array(
        'labels' => array(
            'name' => __('Publications', 'forestt-hub'),
            'singular_name' => __('Publication', 'forestt-hub'),
        ),
        'public' => true,
        'has_archive' => true,
        'supports' => array('title', 'editor', 'excerpt', 'thumbnail'),
        'menu_icon' => 'dashicons-media-document',
        'show_in_rest' => true,
    ));

    // Présentations
    register_post_type('presentations', array(
        'labels' => array(
            'name' => __('Présentations', 'forestt-hub'),
            'singular_name' => __('Présentation', 'forestt-hub'),
        ),
        'public' => true,
        'has_archive' => true,
        'supports' => array('title', 'editor', 'excerpt', 'thumbnail'),
        'menu_icon' => 'dashicons-slides',
        'show_in_rest' => true,
    ));

    // Vidéos
    register_post_type('videos', array(
        'labels' => array(
            'name' => __('Vidéos', 'forestt-hub'),
            'singular_name' => __('Vidéo', 'forestt-hub'),
        ),
        'public' => true,
        'has_archive' => true,
        'supports' => array('title', 'editor', 'excerpt', 'thumbnail'),
        'menu_icon' => 'dashicons-format-video',
        'show_in_rest' => true,
    ));

    // Événements
    register_post_type('evenements', array(
        'labels' => array(
            'name' => __('Événements', 'forestt-hub'),
            'singular_name' => __('Événement', 'forestt-hub'),
        ),
        'public' => true,
        'has_archive' => true,
        'supports' => array('title', 'editor', 'excerpt', 'thumbnail'),
        'menu_icon' => 'dashicons-calendar',
        'show_in_rest' => true,
    ));

    // Téléchargements
    register_post_type('telecharger', array(
        'labels' => array(
            'name' => __('Téléchargements', 'forestt-hub'),
            'singular_name' => __('Téléchargement', 'forestt-hub'),
        ),
        'public' => true,
        'has_archive' => true,
        'supports' => array('title', 'editor', 'excerpt', 'thumbnail'),
        'menu_icon' => 'dashicons-download',
        'show_in_rest' => true,
    ));

    // Partenaires
    register_post_type('partenaires', array(
        'labels' => array(
            'name' => __('Partenaires', 'forestt-hub'),
            'singular_name' => __('Partenaire', 'forestt-hub'),
        ),
        'public' => true,
        'has_archive' => true,
        'supports' => array('title', 'editor', 'excerpt', 'thumbnail'),
        'menu_icon' => 'dashicons-groups',
        'show_in_rest' => true,
    ));
}
add_action('init', 'forestt_register_post_types');

/**
 * Enregistrement des taxonomies
 */
function forestt_register_taxonomies() {
    // Type de ressource
    register_taxonomy('resource_types', array('publications', 'presentations', 'videos', 'telecharger'), array(
        'labels' => array(
            'name' => __('Types de ressource', 'forestt-hub'),
            'singular_name' => __('Type de ressource', 'forestt-hub'),
        ),
        'hierarchical' => true,
        'public' => true,
        'show_in_rest' => true,
    ));

    // Type de partenaire
    register_taxonomy('partner_types', array('partenaires'), array(
        'labels' => array(
            'name' => __('Types de partenaire', 'forestt-hub'),
            'singular_name' => __('Type de partenaire', 'forestt-hub'),
        ),
        'hierarchical' => true,
        'public' => true,
        'show_in_rest' => true,
    ));

    // Catégorie de WP
    register_taxonomy('wp_categories', array('work_packages', 'telecharger', 'publications', 'presentations', 'videos'), array(
        'labels' => array(
            'name' => __('Catégories de WP', 'forestt-hub'),
            'singular_name' => __('Catégorie de WP', 'forestt-hub'),
        ),
        'hierarchical' => true,
        'public' => true,
        'show_in_rest' => true,
    ));
}
add_action('init', 'forestt_register_taxonomies');

/**
 * Ajouter des métadonnées personnalisées à l'API REST
 */
function forestt_add_meta_to_api() {
    $post_types = array('post', 'page', 'work_packages', 'publications', 'presentations', 'videos', 'evenements', 'telecharger', 'partenaires');
    
    foreach ($post_types as $post_type) {
        register_rest_field($post_type, 'meta', array(
            'get_callback' => 'forestt_get_post_meta',
            'schema' => null,
        ));
    }
}
add_action('rest_api_init', 'forestt_add_meta_to_api');

function forestt_get_post_meta($object) {
    $post_id = $object['id'];
    $meta = array(
        'editor_type' => get_post_meta($post_id, '_editor_type', true) ?: 'gutenberg',
        'wp_template' => get_page_template_slug($post_id) ? str_replace('.php', '', str_replace('template-', '', get_page_template_slug($post_id))) : 'default',
    );
    
    return $meta;
}

/**
 * Intégrer les champs ACF dans l'API REST
 */
function forestt_acf_to_rest_api($response, $post, $request) {
    if (!function_exists('get_fields')) return $response;
    
    if (isset($post)) {
        $acf = get_fields($post->ID);
        if (!empty($acf)) {
            $response->data['acf'] = $acf;
        }
    }
    return $response;
}

$post_types = array('post', 'page', 'work_packages', 'publications', 'presentations', 'videos', 'evenements', 'telecharger', 'partenaires');
foreach ($post_types as $post_type) {
    add_filter("rest_prepare_{$post_type}", 'forestt_acf_to_rest_api', 10, 3);
}

/**
 * Ajouter le template dans l'API REST
 */
function forestt_add_template_to_api() {
    register_rest_field('page', 'template', array(
        'get_callback' => 'forestt_get_page_template',
        'schema' => null,
    ));
}
add_action('rest_api_init', 'forestt_add_template_to_api');

function forestt_get_page_template($object) {
    $template = get_page_template_slug($object['id']);
    
    if (!$template) {
        return 'default';
    }
    
    // Supprimer l'extension .php et le préfixe template-
    $template = str_replace('.php', '', $template);
    $template = str_replace('template-', '', $template);
    
    return $template;
}

/**
 * Charger les fichiers ACF pour les champs personnalisés
 */
function forestt_load_acf_fields() {
    if (function_exists('acf_add_local_field_group')) {
        include_once FORESTT_THEME_DIR . '/inc/acf-fields.php';
    }
}
add_action('acf/init', 'forestt_load_acf_fields');

/**
 * Création des modèles de page
 */
function forestt_page_templates($templates) {
    $custom_templates = array(
        'template-about.php' => 'À propos',
        'template-presentation.php' => 'Présentation du programme',
        'template-work-packages.php' => 'Work Packages (WP)',
        'template-partners.php' => 'Partenaires',
        'template-collaborators.php' => 'Chercheurs impliqués',
        'template-resources.php' => 'Ressources',
        'template-deliverables.php' => 'Livrables',
        'template-publications.php' => 'Publications',
        'template-presentations.php' => 'Présentations',
        'template-data.php' => 'Données',
        'template-videos.php' => 'Médias',
        'template-activites.php' => 'Activités',
        'template-colloques.php' => 'Colloques scientifiques',
        'template-doctoral-schools.php' => 'Écoles doctorales',
        'template-living-labs.php' => 'Living Labs forestiers',
        'template-reseaux-formation.php' => 'Réseaux de formation',
        'template-residences.php' => 'Résidences d\'artistes',
        'template-webinaires.php' => 'Webinaires',
        'template-news.php' => 'Actualités',
        'template-contact.php' => 'Contact',
    );
    
    return array_merge($templates, $custom_templates);
}
add_filter('theme_page_templates', 'forestt_page_templates');

// Inclure des fichiers additionnels
require_once FORESTT_THEME_DIR . '/inc/customizer.php';
require_once FORESTT_THEME_DIR . '/inc/template-functions.php';
